<?php
require_once 'config.php';

// Set content type
header('Content-Type: application/json');

// Get request method
$method = $_SERVER['REQUEST_METHOD'];

// Handle different HTTP methods
switch ($method) {
    case 'GET':
        handleGet();
        break;
    case 'POST':
        handlePost();
        break;
    case 'PUT':
        handlePut();
        break;
    case 'DELETE':
        handleDelete();
        break;
    default:
        jsonResponse(false, 'Method not allowed', null, 405);
        break;
}

function handleGet() {
    // Require authentication
    $session = requireAuth();
    
    try {
        $db = getDB();
        
        // Get query parameters
        $month = isset($_GET['month']) ? (int)$_GET['month'] : null;
        $activeOnly = isset($_GET['active_only']) ? (bool)$_GET['active_only'] : false;
        
        // Build query
        $sql = "SELECT * FROM events WHERE 1=1";
        $params = [];
        
        if ($month && $month >= 1 && $month <= 12) {
            $sql .= " AND month = ?";
            $params[] = $month;
        }
        
        if ($activeOnly) {
            $sql .= " AND is_active = 1";
        }
        
        $sql .= " ORDER BY month, day, title";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $events = $stmt->fetchAll();
        
        jsonResponse(true, 'Events retrieved successfully', ['events' => $events]);
        
    } catch (Exception $e) {
        error_log('Get events error: ' . $e->getMessage());
        jsonResponse(false, 'Failed to retrieve events', null, 500);
    }
}

function handlePost() {
    // Require authentication
    $session = requireAuth();
    
    // Get JSON input
    $jsonInput = file_get_contents('php://input');
    $data = json_decode($jsonInput, true);
    
    if (!$data) {
        jsonResponse(false, 'Invalid JSON data', null, 400);
    }
    
    // Sanitize input
    $eventData = [
        'month' => sanitizeInput($data['month'] ?? ''),
        'day' => sanitizeInput($data['day'] ?? ''),
        'title' => sanitizeInput($data['title'] ?? ''),
        'description' => sanitizeInput($data['description'] ?? ''),
        'is_active' => sanitizeInput($data['is_active'] ?? 1)
    ];
    
    // Validate input
    $validationRules = [
        'month' => ['required' => true, 'type' => 'int', 'range' => [1, 12]],
        'day' => ['required' => true, 'type' => 'int', 'range' => [1, 31]],
        'title' => ['required' => true, 'min' => 1, 'max' => 255],
        'description' => ['required' => false, 'max' => 1000],
        'is_active' => ['required' => true, 'type' => 'int', 'range' => [0, 1]]
    ];
    
    $errors = validateInput($eventData, $validationRules);
    
    if (!empty($errors)) {
        jsonResponse(false, 'Validation failed', $errors, 400);
    }
    
    // Additional validation: check if day is valid for the month
    if (!isValidDate($eventData['month'], $eventData['day'])) {
        jsonResponse(false, 'Invalid date for the specified month', null, 400);
    }
    
    try {
        $db = getDB();
        
        // Check for duplicate event
        $checkStmt = $db->prepare("SELECT id FROM events WHERE month = ? AND day = ? AND title = ?");
        $checkStmt->execute([$eventData['month'], $eventData['day'], $eventData['title']]);
        
        if ($checkStmt->fetch()) {
            jsonResponse(false, 'Event with this title already exists on this date', null, 409);
        }
        
        // Insert event
        $stmt = $db->prepare("INSERT INTO events (month, day, title, description, is_active) VALUES (?, ?, ?, ?, ?)");
        $result = $stmt->execute([
            $eventData['month'],
            $eventData['day'],
            $eventData['title'],
            $eventData['description'],
            $eventData['is_active']
        ]);
        
        if ($result) {
            $eventId = $db->lastInsertId();
            jsonResponse(true, 'Event created successfully', ['id' => $eventId], 201);
        } else {
            jsonResponse(false, 'Failed to create event', null, 500);
        }
        
    } catch (Exception $e) {
        error_log('Create event error: ' . $e->getMessage());
        jsonResponse(false, 'Failed to create event', null, 500);
    }
}

function handlePut() {
    // Require authentication
    $session = requireAuth();
    
    // Get event ID from URL
    $eventId = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$eventId) {
        jsonResponse(false, 'Event ID required', null, 400);
    }
    
    // Get JSON input
    $jsonInput = file_get_contents('php://input');
    $data = json_decode($jsonInput, true);
    
    if (!$data) {
        jsonResponse(false, 'Invalid JSON data', null, 400);
    }
    
    // Sanitize input
    $eventData = [
        'month' => sanitizeInput($data['month'] ?? ''),
        'day' => sanitizeInput($data['day'] ?? ''),
        'title' => sanitizeInput($data['title'] ?? ''),
        'description' => sanitizeInput($data['description'] ?? ''),
        'is_active' => sanitizeInput($data['is_active'] ?? '')
    ];
    
    // Validate input
    $validationRules = [
        'month' => ['required' => true, 'type' => 'int', 'range' => [1, 12]],
        'day' => ['required' => true, 'type' => 'int', 'range' => [1, 31]],
        'title' => ['required' => true, 'min' => 1, 'max' => 255],
        'description' => ['required' => false, 'max' => 1000],
        'is_active' => ['required' => true, 'type' => 'int', 'range' => [0, 1]]
    ];
    
    $errors = validateInput($eventData, $validationRules);
    
    if (!empty($errors)) {
        jsonResponse(false, 'Validation failed', $errors, 400);
    }
    
    // Additional validation: check if day is valid for the month
    if (!isValidDate($eventData['month'], $eventData['day'])) {
        jsonResponse(false, 'Invalid date for the specified month', null, 400);
    }
    
    try {
        $db = getDB();
        
        // Check if event exists
        $checkStmt = $db->prepare("SELECT id FROM events WHERE id = ?");
        $checkStmt->execute([$eventId]);
        
        if (!$checkStmt->fetch()) {
            jsonResponse(false, 'Event not found', null, 404);
        }
        
        // Check for duplicate event (excluding current event)
        $duplicateStmt = $db->prepare("SELECT id FROM events WHERE month = ? AND day = ? AND title = ? AND id != ?");
        $duplicateStmt->execute([$eventData['month'], $eventData['day'], $eventData['title'], $eventId]);
        
        if ($duplicateStmt->fetch()) {
            jsonResponse(false, 'Event with this title already exists on this date', null, 409);
        }
        
        // Update event
        $stmt = $db->prepare("UPDATE events SET month = ?, day = ?, title = ?, description = ?, is_active = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
        $result = $stmt->execute([
            $eventData['month'],
            $eventData['day'],
            $eventData['title'],
            $eventData['description'],
            $eventData['is_active'],
            $eventId
        ]);
        
        if ($result) {
            jsonResponse(true, 'Event updated successfully');
        } else {
            jsonResponse(false, 'Failed to update event', null, 500);
        }
        
    } catch (Exception $e) {
        error_log('Update event error: ' . $e->getMessage());
        jsonResponse(false, 'Failed to update event', null, 500);
    }
}

function handleDelete() {
    // Require authentication
    $session = requireAuth();
    
    // Get event ID from URL
    $eventId = isset($_GET['id']) ? (int)$_GET['id'] : null;
    
    if (!$eventId) {
        jsonResponse(false, 'Event ID required', null, 400);
    }
    
    try {
        $db = getDB();
        
        // Check if event exists
        $checkStmt = $db->prepare("SELECT id, title FROM events WHERE id = ?");
        $checkStmt->execute([$eventId]);
        $event = $checkStmt->fetch();
        
        if (!$event) {
            jsonResponse(false, 'Event not found', null, 404);
        }
        
        // Delete event
        $stmt = $db->prepare("DELETE FROM events WHERE id = ?");
        $result = $stmt->execute([$eventId]);
        
        if ($result) {
            jsonResponse(true, 'Event deleted successfully', ['title' => $event['title']]);
        } else {
            jsonResponse(false, 'Failed to delete event', null, 500);
        }
        
    } catch (Exception $e) {
        error_log('Delete event error: ' . $e->getMessage());
        jsonResponse(false, 'Failed to delete event', null, 500);
    }
}

function isValidDate($month, $day) {
    // Basic validation for day-month combinations
    if ($month < 1 || $month > 12 || $day < 1 || $day > 31) {
        return false;
    }
    
    // Check specific month-day combinations
    $daysInMonth = [
        1 => 31, // January
        2 => 29, // February (leap year consideration)
        3 => 31, // March
        4 => 30, // April
        5 => 31, // May
        6 => 30, // June
        7 => 31, // July
        8 => 31, // August
        9 => 30, // September
        10 => 31, // October
        11 => 30, // November
        12 => 31  // December
    ];
    
    return $day <= $daysInMonth[$month];
}
?>